/*********************************************************************
  Copyright:   : Copyright (c) 2002, 2010 MARX(R) CryptoTech LP
  Project      : CBIOSLoader
  Created      : 10Aug2009
  Modified     : 21Apr2010
  Tools        : MSVS2005,2008/MSVS2010
  Description  : Multiplatform CBIOS loader helper
  Platform     : Microsoft .NET Frameword 2.0/4.0
*********************************************************************/

/*********************************************************************
Description:
  CBIOSLoader provides dynamic load of required CBIOS4NET assembly (x86 or x64) 
  depending on the current platform.

  The same source code of CBIOSLoader can be used for both currently available 
  builds of CBIOS4NET assemblies:
	- CBIOS4NET for .NET 2.x SP1, 3.0, 3.5 (MSVS2005, 2008)
	- CBIOS4NET for .NET 4 and later (MSVS2010)
 
Usage:
  1) Prepare your solution
     Place CBIOS4NET.dll & CBIOS4NET64.dll (signed build with/without strong name) 
     assemblies in some directory relative to your finial project binaries,
     for example: bin\asm\
  2) Add reference to one of them (depending on you development platform)
  3) Set reference property "Copy local" to false
  4) Add this file (CBIOSLoader.cs) to your project
  5) Modify your entry point method with initialization of 
     CBIOSLoaderHelper class like this:
 	 static void Main ( )
     {
       CBIOSLoader.CBIOSLoaderHelper.Initialize(Path.GetFullPath("..\\asms"));
       // Your code goes here
     }
     where Initialize method parameter is a path to a directory containing 
     CBIOS4NET*.dll files
  6) Note: The above code must be executed prior to any call to CBIOS4NET. 
     Moreover, it can not be placed to a method conatining any calls 
     to CBIOS4NET.
*********************************************************************/


using System;
using System.Collections.Generic;
using System.Text;
using System.Reflection;
using System.IO;

namespace CBIOSLoader
{
    public static class
        CBIOSLoaderHelper
    {
		class
			AssemblyIdentity
		{
			string _assemblyName;
			string _fileName;

			public AssemblyIdentity
				( string assemblyName, string fileName )
			{
				_assemblyName = assemblyName;
				_fileName = fileName;
			}

			public string AssemblyName
			{
				get
				{
					return _assemblyName;
				}
			}

			public string FileName
			{
				get
				{
					return _fileName;
				}
			}
		}

		class
			MultiplatformAssemblyLoader
		{
			string _assemblyIdentityName;
			AssemblyIdentity _x86AssemblyIdentity;
			AssemblyIdentity _x64AssemblyIdentity;
			string _location;

			public MultiplatformAssemblyLoader
				( string assemblyIdentityName
				, AssemblyIdentity x86AssemblyIdentity
				, AssemblyIdentity x64AssemblyIdentity
				, string location )
			{
				_assemblyIdentityName = assemblyIdentityName;

				_x86AssemblyIdentity = x86AssemblyIdentity;
				_x64AssemblyIdentity = x64AssemblyIdentity;

				_location = location;

				AppDomain.CurrentDomain.AssemblyResolve += new ResolveEventHandler ( CurrentDomain_AssemblyResolve );
			}

			Assembly LoadAssembly ( string requedName )
			{
				AssemblyIdentity platformIdentity;
				if ( IntPtr.Size == 4 )
					platformIdentity = _x86AssemblyIdentity;
				else
					platformIdentity = _x64AssemblyIdentity;

				AssemblyName assemblyName = new AssemblyName ( requedName );
				assemblyName.Name = platformIdentity.AssemblyName;
				assemblyName.CodeBase = platformIdentity.FileName;

				if ( null != _location )
					assemblyName.CodeBase = Path.Combine ( _location, assemblyName.CodeBase );

				Assembly result = Assembly.Load ( assemblyName );
				if ( null == result )
					throw new InvalidProgramException
						( string.Format ( "Failed to load assembly '{0}'"
						, assemblyName ) );
				return result;
			}

			Assembly CurrentDomain_AssemblyResolve ( object sender, ResolveEventArgs args )
			{
				if ( args.Name.StartsWith ( _assemblyIdentityName + ",", StringComparison.InvariantCultureIgnoreCase ) )
				{
					AppDomain.CurrentDomain.AssemblyResolve -= new ResolveEventHandler ( CurrentDomain_AssemblyResolve );

					return LoadAssembly ( args.Name );
				}
				return null;
			}
		}


			private static readonly AssemblyIdentity CBIOSx86Identity = new AssemblyIdentity ( "CBIOS4NET", "CBIOS4NET.dll" );
			private static readonly AssemblyIdentity CBIOSx64Identity = new AssemblyIdentity ( "CBIOS4NET64", "CBIOS4NET64.dll" );

			public static void Initialize ( )
			{
				Initialize ( null );
			}

			public static void Initialize ( string location )
			{
				new MultiplatformAssemblyLoader
				( "CBIOS4NET"
				, CBIOSx86Identity
				, CBIOSx64Identity
				, location );
			}
		
	}
}
